﻿using LocalizationGenerator.Models;
using Microsoft.VisualStudio.TextTemplating;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text.RegularExpressions;

namespace LocalizationGenerator
{
    public abstract class LocalizationGenerator : TextTransformation
    {
        #region Fields & Constants

        private const string DefaultRootClassName = "Resources";
        private const string DefaultRootNamespace = "Localization";

        private readonly Regex _stringParameterExp = new Regex(@"{(?<exp>[^}]+)}", RegexOptions.Compiled);

        private int _indentSpaceCount;
        private string _pathToResource;
        private XmlLocalizationAnalyzer _xmlLocalizationAnalyzer;
        private List<string> _generatedClass;

        #endregion

        #region Methods

        public override string TransformText()
        {
            throw new NotImplementedException();
        }

        public void Generate(string basePath, string pathToResource)
        {
            Generate(basePath, pathToResource, DefaultRootClassName);
        }

        public void Generate(string basePath, string pathToResource, string rootClassName)
        {
            Generate(basePath, pathToResource, GetDefaultNamespace(), rootClassName);
        }

        public void Generate(string basePath, string pathToResource, string rootNamespace, string rootClassName)
        {
            _indentSpaceCount = 0;
            _pathToResource = pathToResource;
            _generatedClass = new List<string>();

            if (!Directory.Exists(basePath))
            {
                WriteLine($"// Directory not found: {basePath}.");
                return;
            }

            _xmlLocalizationAnalyzer = new XmlLocalizationAnalyzer();
            var files = Directory.EnumerateFiles(basePath, "*.xml", SearchOption.TopDirectoryOnly);

            foreach (var filePath in files)
            {
                _xmlLocalizationAnalyzer.Load(filePath);
            }

            Write(
@"// <auto-generated>
//     This code was generated by a tool.
//     Changes to this file may cause incorrect behavior and will be lost if
//     the code is regenerated.
// </auto-generated>

");

            WriteNamespace(rootNamespace, rootClassName);
        }

        private static string GetDefaultNamespace()
        {
            return System.Runtime.Remoting.Messaging.CallContext.LogicalGetData("NamespaceHint") as string ?? DefaultRootNamespace;
        }

        private void IncreaseIndent()
        {
            _indentSpaceCount += 4;
        }

        private void DecreaseIndent()
        {
            _indentSpaceCount -= 4;
        }

        private string GetIndent()
        {
            return new string(' ', _indentSpaceCount);
        }

        private void WriteNamespace(string rootNamespace, string rootClassName)
        {
            var className = GetMemberName(rootClassName);

            WriteLine($"using System.ComponentModel;");
            WriteLine($"using System.Collections.Generic;");
            WriteLine($"using System.Collections.ObjectModel;");
            WriteLine($"using System.Globalization;");
            WriteLine($"using System.IO;");
            WriteLine($"using System.Linq;");
            WriteLine($"using System.Reflection;");
            WriteLine($"using System.Threading;");
            WriteLine($"using System.Windows;");
            WriteLine($"using System.Xml;");
            WriteLine($"");
            WriteLine("#pragma warning disable CS1591 // Missing XML comment for publicly visible type or member");
            WriteLine($"namespace {rootNamespace}");
            WriteLine($"{{");
            IncreaseIndent();
            var indent = GetIndent();

            _generatedClass.Add(className);
            WriteLine($"{indent}[EditorBrowsable(EditorBrowsableState.Never)]");
            WriteLine($"{indent}public static partial class {className}");
            WriteLine($"{indent}{{");

            IncreaseIndent();
            indent = GetIndent();

            WriteLine($"{indent}public static event PropertyChangedEventHandler StaticPropertyChanged;");
            WriteLine($"");
            WriteLine($"{indent}public static void RaisePropertyChanged()");
            WriteLine($"{indent}{{");

            IncreaseIndent();
            indent = GetIndent();

            WriteLine($"{indent}StaticPropertyChanged?.Invoke(null, new PropertyChangedEventArgs(string.Empty));");

            DecreaseIndent();
            indent = GetIndent();

            WriteLine($"{indent}}}");
            WriteLine($"");

            WriteLine($"{indent}public static string[] AvailableCultureIds => new string[] {{ \"{string.Join("\", \"", _xmlLocalizationAnalyzer.AvailableLanguages)}\" }};");
            WriteLine($"");

            WriteClassesOrProperties(_xmlLocalizationAnalyzer.LangStrings, className);

            DecreaseIndent();
            indent = GetIndent();

            WriteLine($"{indent}}}");
            WriteLine($"");

            DecreaseIndent();


            var languageHelper = new StreamReader(typeof(LocalizationGenerator).Assembly.GetManifestResourceStream("LocalizationGenerator.Resources.LocalizationHelper.cs")).ReadToEnd();
            languageHelper = languageHelper.Replace("[Namespace].[Class].AvailableCultureIds", $"{className}.AvailableCultureIds");

            IncreaseIndent();
            IncreaseIndent();
            IncreaseIndent();
            indent = GetIndent();
            var raiseAllPropertyChanged = string.Empty;
            foreach (var classFullName in _generatedClass)
            {
                raiseAllPropertyChanged += $"\n{indent}{classFullName}.RaisePropertyChanged();";
            }
            DecreaseIndent();
            DecreaseIndent();
            DecreaseIndent();
            indent = GetIndent();

            languageHelper = languageHelper.Replace("[RaiseAllPropertyChanged]", raiseAllPropertyChanged);

            WriteLine(languageHelper);
            WriteLine($"}}");
            WriteLine("#pragma warning restore CS1591 // Missing XML comment for publicly visible type or member");
        }

        private void WriteClassesOrProperties(List<LangString> langStrings, string parentClassName)
        {
            foreach (var langString in langStrings)
            {
                if (langString.Strings.Count > 0)
                {
                    WriteProperty(langString);
                }
                else if (langString.LangStrings.Count > 0)
                {
                    WriteClass(langString, parentClassName);
                }
            }
        }

        private void WriteClass(LangString langString, string parentClassName)
        {
            var indent = GetIndent();
            var className = GetMemberName(langString.XPath);

            _generatedClass.Add($"{parentClassName}.{className}");
            WriteLine($"{indent}[EditorBrowsable(EditorBrowsableState.Never)]");
            WriteLine($"{indent}public static partial class {className}");
            WriteLine($"{indent}{{");

            IncreaseIndent();
            indent = GetIndent();

            WriteLine($"{indent}public static event PropertyChangedEventHandler StaticPropertyChanged;");
            WriteLine($"");
            WriteLine($"{indent}public static void RaisePropertyChanged()");
            WriteLine($"{indent}{{");

            IncreaseIndent();
            indent = GetIndent();

            WriteLine($"{indent}StaticPropertyChanged?.Invoke(null, new PropertyChangedEventArgs(string.Empty));");

            DecreaseIndent();
            indent = GetIndent();

            WriteLine($"{indent}}}");
            WriteLine($"");

            WriteClassesOrProperties(langString.LangStrings, $"{parentClassName}.{className}");

            DecreaseIndent();
            indent = GetIndent();

            WriteLine($"{indent}}}");
            WriteLine($"");
        }

        private void WriteProperty(LangString langString)
        {
            var indent = GetIndent();
            var propertyName = GetMemberName(langString.XPath);
            var hasParameters = false;
            var parameters = new List<string>();

            WriteLine($"{indent}/// <summary>");
            foreach (var value in langString.Strings)
            {
                var literalValue = FormatLiteralString(value.Value);

                var matches = _stringParameterExp.Matches(literalValue);
                if (matches.Count > 0)
                {
                    hasParameters = true;
                    foreach (Match match in matches)
                    {
                        var param = match.Value.TrimStart('{').TrimEnd('}');
                        if (!parameters.Contains(param))
                        {
                            parameters.Add(match.Value.TrimStart('{').TrimEnd('}'));
                        }
                    }
                }

                WriteLine($"{indent}/// <para>{value.LanguageId} : {literalValue}</para>");
            }
            WriteLine($"{indent}/// </summary>");
            WriteLine($"{indent}public static string {propertyName} => LocalizationHelper.GetString(\"{_pathToResource}.{langString.Strings.First().FileNameNoExtension}\", \"{langString.XPath}\");");
            WriteLine($"");

            if (hasParameters)
            {
                var parametersString = string.Join(", ", parameters.Select(s => $"object @{s.ToLowerInvariant()}").ToArray());
                var replaceParametersString = string.Empty;
                foreach (var param in parameters)
                {
                    replaceParametersString += $".Replace(\"{{{param}}}\", @{param.ToLowerInvariant()}?.ToString())";
                }
                WriteLine($"{indent}public static string Formatted{propertyName.Trim('@')}({parametersString}) {{ return {propertyName}{replaceParametersString}; }}");
                WriteLine($"");
            }
        }

        private string GetMemberName(string xpath)
        {
            return $"@{xpath.Split('/').Last()}";
        }

        private string FormatLiteralString(string input)
        {
            return input.Replace("\\", "\\\\").Replace("\"", "\\\"").Replace("\r", "\\r").Replace("\n", "\\n");
        }

        #endregion
    }
}
