'use strict';

// Service utilisé pour accéder à l'API REST des chemins
angular.module('mean.application').factory('TrackGenerator', ['Cesium',
function (Cesium) {

  var TrackGenerator = {};
  
  // Conversion depuis tableau de coordonnées géographiques vers GeoJSON
  TrackGenerator.cartographicToGeoJson = function(waypoints) {
	var geoJson = {
		"type": "Feature",
        "geometry": {
			"type": "LineString",
			"coordinates": []
		}
	};
	
	for (var i = 0; i < waypoints.length / 3; i++) {
		geoJson.geometry.coordinates[i] = [ waypoints[3*i], waypoints[3*i+1], waypoints[3*i+2] ];
	}
	
	return geoJson;
  }
  
  // Conversion depuis GeoJSON vers tableau de coordonnées géographiques
  TrackGenerator.geoJsonToCartographic = function(geoJson) {
	var waypoints = [];
	
	for (var i = 0; i < geoJson.geometry.coordinates.length; i++) {
		waypoints[3*i] = geoJson.geometry.coordinates[i][0];
		waypoints[3*i+1] = geoJson.geometry.coordinates[i][1];
		if ( geoJson.geometry.coordinates[i].length >=3 ) {
			waypoints[3*i+2] = geoJson.geometry.coordinates[i][2];
		}
	}
	
	return waypoints;
  }
  
  // Conversion depuis tableau de coordonnées cartésiennes vers géographiques
  TrackGenerator.cartesianToCartographic = function(waypoints) {
	for (var i = 0; i < waypoints.length / 3; i++) {
		var position = new Cesium.Cartesian3(waypoints[3*i], waypoints[3*i+1], waypoints[3*i+2]);
		var cartographicPosition = new Cesium.Cartographic();
		Cesium.Ellipsoid.WGS84.cartesianToCartographic(position, cartographicPosition);
	
		waypoints[3*i] = Cesium.Math.toDegrees(cartographicPosition.longitude);
		waypoints[3*i+1] = Cesium.Math.toDegrees(cartographicPosition.latitude);
		waypoints[3*i+2] = cartographicPosition.height;
	}
  }
  
  // Conversion depuis tableau de coordonnées géographiques vers cartésiennes
  TrackGenerator.cartographicToCartesian = function(waypoints) {
	for (var i = 0; i < waypoints.length / 3; i++) {
		var cartographicPosition = Cesium.Cartographic.fromDegrees(waypoints[3*i], waypoints[3*i+1], waypoints[3*i+2]);
		var position = new Cesium.Cartesian3();
		Cesium.Ellipsoid.WGS84.cartographicToCartesian(cartographicPosition, position);
	
		waypoints[3*i] = position.x;
		waypoints[3*i+1] = position.y;
		waypoints[3*i+2] = position.z;
	}
  }
  
  // Generate the GeoJson layer for the given plain array of cartographic coordinates
  TrackGenerator.generateGeoJson = function(waypoints) {
	return this.cartographicToGeoJson(waypoints);
  }
  
  // Génère une animation 3D au format CZML à partir d'un chemin en coordonnées cartographiques
  TrackGenerator.generateCzml = function(waypoints) {
	// La scène CZML contient :
	// - l'icône qui représentera le véhicule (nommé 'VehicleIcon')
	// - le chemin suivi par le véhicule et utilisé pour interpoler sa position (nommé 'Vehicle')
	// Le chemin est donné comme un tableau de coordonnées cartographiques, donc sans référence par rapport au sol (i.e. altitude).
	// La position réelle au niveau du sol sera calculée en temps-réel lors de l'animation.
	var builtInCzml = 
	[
	  {
		"id":"document",
		"version":"1.0"
	  },
	  {
		"id":"VehicleIcon",
		"billboard":{
		  "eyeOffset":{
			"cartesian":[
			  0.0
			]
		  },
		  "horizontalOrigin":"CENTER",
		  "image":"data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAYAAADgdz34AAAAAXNSR0IArs4c6QAAAARnQU1BAACxjwv8YQUAAAAJcEhZcwAADsMAAA7DAcdvqGQAAAEISURBVEhLvVXBDYQwDOuojHKj8LhBbpTbpBCEkZsmIVTXq1RVQGrHiWlLmTTqPiZBlyLgy/KSZQ5JSHDQ/mCYCsC8106kDU0AdwRnvYZArWRcAl0dcYJq1hWCb3hBrumbDAVMwAC82WoRvgMnVMDBnB0nYZFTbE6BBvdUGqVqCbjBIk3PyFFR/NU7EKzru+qZsau3ryPwwCRLKYOzutZuCL6fUmWeJGzNzL/RxAMrUmASSCkkAayk2IxPlwhAAYGpsiHQjbLccfdOY5gKkCXAMi7SscAwbQpAnKyctWyUZ6z8ja3OGMepwD8asz+9FnSvbhU8uVOHFIwQsI3/p0CfhuqCSQuxLqsN6mu8SS+N42MAAAAASUVORK5CYII=",
		  "pixelOffset":{
			"cartesian2":[
			  0.0,0.0
			]
		  },
		  "scale":0.8333333333333334,
		  "show":[
			{
			  "boolean":true
			}
		  ],
		  "verticalOrigin":"BOTTOM"
		},
		"label":{
		  "fillColor":[
			{
			  "rgba":[
				255,255,0,255
			  ]
			}
		  ],
		  "font":"bold 12pt Segoe UI Semibold",
		  "horizontalOrigin":"LEFT",
		  "outlineColor":{
			"rgba":[
			  0,0,0,255
			]
		  },
		  "pixelOffset":{
			"cartesian2":[
			  16,-32.0
			]
		  },
		  "scale":1.0,
		  "show":[
			{
			  "boolean":true
			}
		  ],
		  "style":"FILL",
		  "text":"Vous",
		  "verticalOrigin":"CENTER"
		} },
	  {
		"id":"Vehicle",
		"availability":"",
		"path":{
		  "material":{
			  "solidColor":{
				"color":{
					"rgba":[
					  255,255,0,255
					]
				  }
				}
		  },
		  "width":[
			{
			  "number":5.0
			}
		  ],
		  "show":[
			{
			  "boolean":false // Will be visible when it will be possible to clamp it on ground
			}
		  ]
		},
		"position":{
		  "interpolationAlgorithm":"LAGRANGE",
		  "interpolationDegree":1,
		  "epoch":"",
		  "cartesian":[]
		}
	  }
	];
	
	var cartesianRoute = waypoints.slice();
	this.cartographicToCartesian( cartesianRoute );
	
	// Définit la date et l'heure de départ
	var epoch = Cesium.JulianDate.now();
	var previousPosition;
	var timeStep = 0;
	
	// L'animation 3D est définie sous la forme d'une liste de temps/position
	for (var i = 0; i < waypoints.length / 3; i++) {
		var position = new Cesium.Cartesian3(cartesianRoute[3*i], cartesianRoute[3*i+1], cartesianRoute[3*i+2]);
		
		if ( Cesium.defined( previousPosition ) )
		{
			var distance = Cesium.Cartesian3.distance(previousPosition, position);
			// Calcul du pas de temps, on considère une vitesse de 90 km/h soit environ 25 m/s
			timeStep += distance / 25;
		}
		// Pas de temps
		builtInCzml[2].position.cartesian[4*i] = timeStep;
		// Position
		builtInCzml[2].position.cartesian[4*i+1] = position.x;
		builtInCzml[2].position.cartesian[4*i+2] = position.y;
		builtInCzml[2].position.cartesian[4*i+3] = position.z;
		
		previousPosition = position;
	}
	
	// Calcul de la date et l'heure de fin
	var end = new Cesium.JulianDate();
	Cesium.JulianDate.addSeconds(epoch, timeStep, end);
	var timeInterval = new Cesium.TimeInterval({
		start : epoch,
		stop : end,
		isStartTimeIncluded : true,
		isStopTimeIncluded : true
	});
	builtInCzml[2].position.epoch = epoch.toString();
	builtInCzml[2].availability = timeInterval.toString();
	
	return builtInCzml;
  };
  
  return TrackGenerator;
  
}]);
