type token =
  | Token_add | Token_sub | Token_mul | Token_div | Token_mod
  | Token_pal | Token_par
  | Token_number of char list

type tree =
  | Add of tree * tree
  | Sub of tree * tree
  | Mul of tree * tree
  | Div of tree * tree
  | Mod of tree * tree
  | Number of int

let lexing str =
  let ( >>= ) (cur_token, old_token, tokens) func = match old_token with
    | Some old_token -> func None (cur_token :: old_token :: tokens)
    | None -> func None (cur_token :: tokens)
  in let ( >|= ) (digit, old_token, tokens) func = match old_token with
    | Some (Token_number l) -> func (Some (Token_number (digit :: l))) tokens
    | Some old_token -> func (Some (Token_number [digit])) (old_token :: tokens)
    | None -> func (Some (Token_number [digit])) tokens
  in let rec aux old_token tokens idx =
    if String.length str = idx
    then List.rev @@ match old_token with
      | Some old_token -> old_token :: tokens
      | None -> tokens
    else match str.[idx] with
      | '+' -> ((Token_add, old_token, tokens) >>= aux) @@ (idx + 1)
      | '-' -> ((Token_sub, old_token, tokens) >>= aux) @@ (idx + 1)
      | '*' -> ((Token_mul, old_token, tokens) >>= aux) @@ (idx + 1)
      | '/' -> ((Token_div, old_token, tokens) >>= aux) @@ (idx + 1)
      | '%' -> ((Token_mod, old_token, tokens) >>= aux) @@ (idx + 1)
      | '(' -> ((Token_par, old_token, tokens) >>= aux) @@ (idx + 1)
      | ')' -> ((Token_pal, old_token, tokens) >>= aux) @@ (idx + 1)
      | '0' .. '9' as digit -> ((digit, old_token, tokens) >|= aux) @@ (idx + 1)
      | _ -> match old_token with
        | Some token -> aux None (token :: tokens) (idx + 1)
        | None -> aux None tokens (idx + 1)
  in aux None [] 0

let to_number lst =
  Number (List.fold_left (fun acc x -> acc * 10 + (int_of_char x - 48)) 0 (List.rev lst))

let to_operator operator a b = match operator with
  | Token_add -> Add (a, b)
  | Token_sub -> Sub (a, b)
  | Token_mul -> Mul (a, b)
  | Token_div -> Div (a, b)
  | Token_mod -> Mod (a, b)
  | _ -> raise (Failure "Bad expression")

let p = function
  | Token_add -> 5
  | Token_sub -> 5
  | Token_mul -> 10
  | Token_div -> 10
  | Token_mod -> 10
  | Token_pal -> 0
  | Token_par -> 0
  | _ -> raise (Failure "Bad expression")

let parsing tokens =
  let node operator = function
    | b :: a :: r -> (to_operator operator a b) :: r
    | _ -> raise (Failure "Bad expression")
  in let rec unstack o1 stack tree = match o1, stack with
    | Some Token_pal, Token_par :: r -> r, tree
    | Some Token_pal, o2 :: r -> unstack (Some Token_pal) r (node o2 tree)
    | Some o1, o2 :: r when (p o1) < (p o2) -> unstack (Some o1) r (node o2 tree)
    | None, o2 :: r -> unstack None r (node o2 tree)
    | _ -> (stack, tree)
  in let rec aux stack tree = function
    | [] -> snd @@ unstack None stack tree
    | (Token_add | Token_sub | Token_mul | Token_div | Token_mod) as o1 :: r ->
      let (new_stack, new_tree) = unstack (Some o1) stack tree in
      aux (o1 :: new_stack) new_tree r
    | Token_par :: r -> aux (Token_par :: stack) tree r
    | Token_pal :: r ->
      let (new_stack, new_tree) = unstack (Some Token_pal) stack tree in
      aux new_stack new_tree r
    | (Token_number l) :: r -> aux stack ((to_number l) :: tree) r
  in match aux [] [] tokens with
    | [ tree ] -> tree
    | _ -> raise (Failure "Bad expression")

let rec eval = function
  | Add (a, b) -> ( + ) (eval a) (eval b)
  | Sub (a, b) -> ( - ) (eval a) (eval b)
  | Mul (a, b) -> ( * ) (eval a) (eval b)
  | Div (a, b) -> ( / ) (eval a) (eval b)
  | Mod (a, b) -> ( mod ) (eval a) (eval b)
  | Number i -> i

let () = 
  if Array.length Sys.argv > 1
  then
    try Printf.printf "%d\n" (eval @@ parsing @@ lexing Sys.argv.(1))
    with Failure s -> print_endline s
  else
    print_endline (Sys.argv.(0) ^ " expression")
