package com.annotation;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Type;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.context.annotation.ClassPathScanningCandidateComponentProvider;
import org.springframework.core.type.filter.AssignableTypeFilter;

public class ExportedFieldsFinder {
	private Map<Class<?>, List<FieldDescriptorExportable>> classes;
	private Map<String, FieldDescriptorExportable> findCache;
	private List<Class<?>> knownClasses;
	public ExportedFieldsFinder(String packageName) {
		classes = new HashMap<Class<?>, List<FieldDescriptorExportable>>();
		findCache = new HashMap<String, FieldDescriptorExportable>();
		knownClasses = new LinkedList<Class<?>>();
		
		ClassPathScanningCandidateComponentProvider scanner = new ClassPathScanningCandidateComponentProvider(true);
		scanner.addIncludeFilter(new AssignableTypeFilter(ExportableRootClass.class));
		for(BeanDefinition bean : scanner.findCandidateComponents(packageName)) {
			try {
				Class<?> clazz = Class.forName(bean.getBeanClassName());
				for(Type type : clazz.getGenericInterfaces()) {
					if(type instanceof Class<?>) {
						if(((Class<?>) type).isAssignableFrom(ExportableRootClass.class)) {
							knownClasses.add(clazz);
							break;
						}
					}
				}
			} catch (ClassNotFoundException e) {
				e.printStackTrace();
			}
		}
		for(Class<?> clazz : knownClasses) {
			classes.put(clazz, populateClass(clazz, clazz, null, true));
		}
	}
	public ExportedFieldsFinder() {
		this("");
	}
	/**
	 * @param c Class type to look for. Must be listed in @enum AnalysisType
	 * @return All exportable fields found in class and linked-classes recursively
	 */
	public List<FieldDescriptorExportable> get(Class<?> c) {
		return classes.get(c);
	}
	public FieldDescriptorExportable findField(String uid) {
		return findCache.get(uid);
	}
	private List<FieldDescriptorExportable> populateClass(Class<?> c, Class<?> masterClass, FieldDescriptor parent, boolean addToCache) {
		List<FieldDescriptorExportable> fields = new LinkedList<FieldDescriptorExportable>();
		if(c.getSuperclass() != null)
			fields.addAll(populateClass(c.getSuperclass(), masterClass, parent, addToCache));
		for(Field field : c.getDeclaredFields()) {
			if(field.isAnnotationPresent(ExportField.class)) {
				ExportField ann = field.getAnnotation(ExportField.class);
				FieldDescriptorExportable newField = new FieldDescriptorExportable(masterClass, field, parent, ann);
				fields.add(newField);
				if(addToCache)
					findCache.put(newField.getUid(), newField);
			} 
			if(field.isAnnotationPresent(ExportClassField.class)) {
				if(Collection.class.isAssignableFrom(field.getType())) {
					try {
						fields.addAll(populateClass(Class.forName(field.getAnnotation(ExportClassField.class).className()), masterClass, new FieldDescriptor(masterClass, field, parent), addToCache));
					} catch (ClassNotFoundException e) {
						e.printStackTrace();
					}
				} else {
					fields.addAll(populateClass(field.getType(), masterClass, new FieldDescriptor(masterClass, field, parent), addToCache));
				}
			} 
			if(field.isAnnotationPresent(ExportClassCollectionField.class)) {
				ExportClassCollectionField annotation = field.getAnnotation(ExportClassCollectionField.class);
				FieldDescriptorCollectionExportable descriptor = new FieldDescriptorCollectionExportable(masterClass, field, parent, annotation);
				fields.add(descriptor);
				if(addToCache)
					findCache.put(descriptor.getUid(), descriptor);
				try {
					descriptor.setChildren(populateClass(Class.forName(annotation.className()), masterClass, descriptor, false));
				} catch (ClassNotFoundException e) {
					e.printStackTrace();
				}
			}
		}
		for(Method method : c.getDeclaredMethods()) {
			if(method.isAnnotationPresent(ExportField.class)) {
				ExportField ann = method.getAnnotation(ExportField.class);
				FieldDescriptorExportable newField = new FieldDescriptorExportable(masterClass, method, parent, ann);
				fields.add(newField);
				if(addToCache)
					findCache.put(newField.getUid(), newField);
			} 
			if(method.isAnnotationPresent(ExportClassField.class)) {
				if(Collection.class.isAssignableFrom(method.getReturnType())) {
					try {
						fields.addAll(populateClass(Class.forName(method.getAnnotation(ExportClassField.class).className()), masterClass, new FieldDescriptor(masterClass, method, parent), addToCache));
					} catch (ClassNotFoundException e) {
						e.printStackTrace();
					}
				} else {
					fields.addAll(populateClass(method.getReturnType(), masterClass, new FieldDescriptor(masterClass, method, parent), addToCache));
				}
			} 
			if(method.isAnnotationPresent(ExportClassCollectionField.class)) {
				ExportClassCollectionField annotation = method.getAnnotation(ExportClassCollectionField.class);
				FieldDescriptorCollectionExportable descriptor = new FieldDescriptorCollectionExportable(masterClass, method, parent, annotation);
				fields.add(descriptor);
				if(addToCache)
					findCache.put(descriptor.getUid(), descriptor);
				try {
					descriptor.setChildren(populateClass(Class.forName(annotation.className()), masterClass, descriptor, false));
				} catch (ClassNotFoundException e) {
					e.printStackTrace();
				}
			}
		}
		return fields;
	}
	/**
	 * @return the knownClasses
	 */
	public List<Class<?>> getKnownClasses() {
		return knownClasses;
	}
}
