'use strict';

// Contrôleur utilisé pour afficher un chemin sur un globe 3D
angular.module('mean.application').controller('TrackGlobeController', ['$scope', '$http', '$stateParams', 'TrackService', 'TrackGenerator',
  function($scope, $http, $stateParams, TrackService, TrackGenerator) {
    // Projète au sol la position donnée
    var clampToGround = function(position) {
      // Position au sol à altitude 0
      var cartographicPosition = new Cesium.Cartographic();
      Cesium.Ellipsoid.WGS84.cartesianToCartographic(position, cartographicPosition);
      // Origine du rayon à la même position mais au-dessus de tous les sommets existants
      var cartographicOrigin = Cesium.Cartographic.clone(cartographicPosition);
      cartographicOrigin.height = 20000; // Everest ~8000m
      // Calcul du point 3D origine
      var origin = new Cesium.Cartesian3();
      Cesium.Ellipsoid.WGS84.cartographicToCartesian(cartographicOrigin, origin);
      // Calcul de la direction du rayon : de l'origine vers le centre de la Terre
      var direction = new Cesium.Cartesian3();
      Cesium.Cartesian3.subtract(position, origin, direction);
      Cesium.Cartesian3.normalize(direction, direction);
      var ray = new Cesium.Ray(origin, direction);
      
      var groundPosition = new Cesium.Cartesian3();
      // Intersection avec le sol
      return $scope.viewer.scene.globe.pick(ray, $scope.viewer.scene, groundPosition);
    }

    // Recherche l'entité dans le contenu 3D
    var getTrackedEntity = function() {
      if (Cesium.defined($scope.lookAt)) {
        $scope.trackedEntity = $scope.viewer.dataSources.get(0).entities.getById($scope.lookAt);
        if (Cesium.defined($scope.trackedEntity)) {
          // Vitesse de défilement par défaut
          $scope.viewer.clock.multiplier = 1;
          // Récupérer l'icône représentant l'entité
          $scope.trackedEntityIcon = $scope.viewer.dataSources.get(0).entities.getById($scope.lookAt + 'Icon');
        }
      }
    }

    // Charger l'objet source au format Czml
    // L'argument lookAt est le nom de l'entité à suivre dans les données source
    var loadSource = function(source, lookAt) {
      $scope.lookAt = lookAt;
      // Chargement de l'animation 3D
      var dataSource = new Cesium.CzmlDataSource();
      dataSource.load(source, 'Built-in CZML');
      $scope.viewer.dataSources.add(dataSource);
      // Récupération de l'entité suivant le chemin
      getTrackedEntity();
      // Positionnement du callback appelé à chaque pas d'exécution de l'animation
      $scope.viewer.clock.onTick.addEventListener(function(clock) {
        if ( Cesium.defined($scope.trackedEntity) ) {
          // Récupération de la position courante
          var cartographicPosition = new Cesium.Cartographic();
          var position = $scope.trackedEntity.position.getValue(clock.currentTime);
          Cesium.Ellipsoid.WGS84.cartesianToCartographic(position, cartographicPosition);
          
          // Calcul de la position projetée au sol
          var groundPosition = clampToGround(position);
          if ( Cesium.defined(groundPosition) ) {
            // Mise à jour de la position 3D en conséquence
            position = groundPosition;
            Cesium.Ellipsoid.WGS84.cartesianToCartographic(groundPosition, cartographicPosition);
            // Et positionnement de l'icone 
            if ( Cesium.defined($scope.trackedEntityIcon) ) {
              $scope.trackedEntityIcon.position = new Cesium.ConstantPositionProperty(groundPosition, Cesium.ReferenceFrame.FIXED);
              var text = cartographicPosition.height.toFixed() + ' m';
              $scope.trackedEntityIcon._label._text = new Cesium.ConstantProperty(text);
            }
          }
          
          // Calcul du repère local à la position
          var transform = Cesium.Transforms.eastNorthUpToFixedFrame(position);
          // Attachement du point de vue à ce repère
          Cesium.Matrix4.clone(transform, $scope.viewer.scene.camera.transform);
          $scope.viewer.scene.camera.constrainedAxis = Cesium.Cartesian3.UNIT_Z;
        }
      });
    }

    //Récupère un chemin via son ID
    $scope.findOne = function() {
      TrackService.get({
        trackId: $stateParams.trackId
      }, function(track) {
        $scope.track = track;

        // Position par défaut de la caméra
        $scope.viewer.scene.camera.lookAt(
          new Cesium.Cartesian3(-1500.0, -1500.0, 500.0),
          Cesium.Cartesian3.ZERO,
          Cesium.Cartesian3.UNIT_Z);
        loadSource( TrackGenerator.generateCzml($scope.track.waypoints), "Vehicle");
      });
    };
  }
]).directive("cesium", ['Cesium', function (Cesium) {
  return {
    restrict: "E",
    controllerAs: "TrackGlobeController",
    link: function (scope, element, attributes) {
      var options = {
        sceneMode : Cesium.SceneMode.SCENE3D,
        sceneModePicker : false,
        scene3DOnly : true,
        homeButton : false,
        geocoder : false,
        navigationHelpButton : true,
        baseLayerPicker : false
      }
      scope.viewer = new Cesium.Viewer(element[0], options);
      // Définition des fournisseurs de données image/terrain à utiliser
      scope.viewer.imageProvider = new Cesium.BingMapsImageryProvider({
          url : '//dev.virtualearth.net'
      });
      scope.viewer.terrainProvider = new Cesium.CesiumTerrainProvider({
          url : '//assets.agi.com/stk-terrain/world'
      });
    }
  }
}]);

