'use strict';

var mongoose = require('mongoose'),
    Schema = mongoose.Schema,
    togeojson = require('togeojson'),
    jsdom = require('jsdom').jsdom;

// Déclaration du schéma du modèle 'Track'
var TrackSchema = new mongoose.Schema({
  // Utilisateur ayant créé le chemin
  user : {
  	type: mongoose.Schema.Types.ObjectId,
  	ref: 'User' // Référence le modèle d'utilisateur de MEAN.IO
  },
  // Date de création
  created: {
    type: Date,
    default: Date.now
  },
  // Titre du chemin
  title: {
    type: String,
    required: true,
    trim: true
  },
  // Description associée au chemin
  description : {
  	type : String, required : true
  },
  // Liste des points de passage constituant le chemin (coordonnées géographiques)
  waypoints : {
  	type : [Number], required : false
  },
  // Boite englobante des points (coordonnées géographiques également)
  bbox : {
    type : [Number], required : false
  }
});
// Ajout d'une fonction de validation pour le titre et la description
TrackSchema.path('title').validate(function(title) {
  return !!title;
}, 'Title cannot be blank');
TrackSchema.path('description').validate(function(description) {
  return !!description;
}, 'Description cannot be blank');
// Méthode utilisée pour récupérer un chemin via son ID,
// va récupérer certaines des informations de l'utilisateur via populate
TrackSchema.statics.getById = function(id, cb) {
  this.findOne({
    _id: id
  }).populate('user', 'name username').exec(cb);
};

// Setter permettant de remplir le chemin à partir de données au format KML
TrackSchema.virtual('kml')
.set(function (data) {
  if ( data ) {
    var kml = jsdom(data);
    var geojson = togeojson.kml(kml);
    this.set( 'geojson', geojson );
  }
});

// Setter permettant de remplir le chemin à partir de données au format GPX
TrackSchema.virtual('gpx')
.set(function (data) {
  if ( data ) {
    var gpx = jsdom(data);
    var geojson = togeojson.gpx(gpx);
    this.set( 'geojson', geojson );
  }
});

// Getter permettant de récupérer le chemin au format GeoJSON
TrackSchema.virtual('geojson')
.get(function () {
  var coordinates = [];
  for (var i = 0; i < this.waypoints.length / 3; i++) {
    coordinates[i] = [ this.waypoints[3*i], this.waypoints[3*i+1], this.waypoints[3*i+2] ];
  }
  
  var feature = {
    "type": "Feature",
    "geometry": {
      "type": "LineString",
      "coordinates": coordinates,
    },
    bbox: this.bbox
  };
  
  return feature;
})
// Setter permettant de remplir le chemin à partir de données au format GeoJSON
.set(function (geojson) {
  if ( geojson ) {
    // Récupération de la géométrie
    var geometry = undefined;
    if ( geojson.type == 'LineString' ) {
      geometry = geojson;
    }
    // Si incluse dans une feature 
    else if ( geojson.type == 'Feature' ) {
      geometry = geojson.geometry;
    } else if ( ( geojson.type == 'FeatureCollection' ) && ( geojson.features.length > 0 ) ) {
      geometry = geojson.features[0].geometry;
    }
    
    if ( !geometry) {
      console.log("Unsupported GeoJSON to create a route (require a standalone Feature with a LineString geometry or included a FeatureCollection at first position)");
    }
    
    // Récupération des coordonnées
    var coordinates = undefined;
    if ( geometry.type == 'LineString' ) {
      coordinates = geometry.coordinates;
    }
    
    if ( !coordinates) {
      console.log("Unsupported GeoJSON geometry to create a route (require a LineString)");
    }
    
    // Transformation dans le formalisme de la base de données
    var waypoints = [];
    // Calcul de la boite englobante
    var minLon = 360; var maxLon = -360;
    var minLat = 90; var maxLat = -90;
  
    for (var i = 0; i < coordinates.length; i++) {
      waypoints[3*i] = coordinates[i][0];
      waypoints[3*i+1] = coordinates[i][1];
      // Mise à jour des valeurs min/max
      minLon = Math.min(minLon, waypoints[3*i]);
      minLat = Math.min(minLat, waypoints[3*i+1]);
      maxLon = Math.max(maxLon, waypoints[3*i]);
      maxLat = Math.max(maxLat, waypoints[3*i+1]);
      // Nous nous assurons d'avoir des coordonnées 3D
      if ( coordinates[i].length >= 3 ) {
        waypoints[3*i+2] = coordinates[i][2];
      } else {
        waypoints[3*i+2] = 0;
      }
    }
    
    this.set( 'waypoints', waypoints );
    this.set( 'bbox', [minLon, minLat, maxLon, maxLat] );
  }
});

mongoose.model('Track', TrackSchema);

