using System;
using System.Collections.Generic;
using System.ComponentModel;
using Microsoft.Ccr.Core;
using Microsoft.Dss.Core.Attributes;
using Microsoft.Dss.ServiceModel.Dssp;
using Microsoft.Dss.ServiceModel.DsspServiceBase;
using W3C.Soap;
using submgr = Microsoft.Dss.Services.SubscriptionManager;
using webcam = Microsoft.Robotics.Services.WebCam.Proxy;
using System.Drawing.Imaging;
using System.Windows.Media.Imaging;
using System.Drawing;
using System.IO;

namespace MotionDetector
{
    [Contract(Contract.Identifier)]
    [DisplayName("MotionDetector")]
    [Description("MotionDetector service (no description provided)")]
    class MotionDetectorService : DsspServiceBase
    {
        byte[] grayImgOld, grayImg, motionImg;

        /// <summary>
        /// Service state
        /// </summary>
        [ServiceState]
        MotionDetectorState _state = new MotionDetectorState();

        /// <summary>
        /// Main service port
        /// </summary>
        [ServicePort("/MotionDetector", AllowMultipleInstances = true)]
        MotionDetectorOperations _mainPort = new MotionDetectorOperations();

        [Partner("/WebcamService", Contract = webcam.Contract.Identifier, CreationPolicy = PartnerCreationPolicy.UseExistingOrCreate)]
        webcam.WebCamOperations _webcamServicePort = new Microsoft.Robotics.Services.WebCam.Proxy.WebCamOperations();
        webcam.WebCamOperations _webcamServiceNotify = new Microsoft.Robotics.Services.WebCam.Proxy.WebCamOperations();
   
        [SubscriptionManagerPartner]
        submgr.SubscriptionManagerPort _submgrPort = new submgr.SubscriptionManagerPort();

        /// <summary>
        /// Service constructor
        /// </summary>
        public MotionDetectorService(DsspServiceCreationPort creationPort)
            : base(creationPort)
        {
        }

        /// <summary>
        /// Service start
        /// </summary>
        protected override void Start()
        {

            // 
            // Add service specific initialization here
            // 
            _state.Enabled = false;
            SpawnIterator(Initialize);
        }

        IEnumerator<ITask> Initialize()
        {
            #region Subscribe to webcam notifications
            var subscribe = _webcamServicePort.Subscribe(_webcamServiceNotify, typeof(webcam.UpdateFrame));
            yield return (Choice)subscribe;

            var fault = (Fault)subscribe;
            if (fault != null)
            {
                LogError(fault);
                StartFailed();
                yield break;
            }
            #endregion
            
            StartComplete();
        }

        void StartComplete()
        {
            base.Start();

            MainPortInterleave.CombineWith(
                Arbiter.Interleave(
                    new TeardownReceiverGroup(),
                    new ExclusiveReceiverGroup(
                        Arbiter.Receive<webcam.UpdateFrame>(true, _webcamServiceNotify, UpdateFrameHandler)
                        ),
                    new ConcurrentReceiverGroup()
                )
            );
        }

        public void UpdateFrameHandler(webcam.UpdateFrame update)
        {
            if(_state.Enabled)
                SpawnIterator(update, UpdateFrameHandlerIterator);
        }


        public IEnumerator<ITask> UpdateFrameHandlerIterator(webcam.UpdateFrame update)
        {
            
            Fault fault = null;
            webcam.QueryFrameRequest request = new webcam.QueryFrameRequest();
            webcam.QueryFrameResponse response = null;

            yield return Arbiter.Choice(
                _webcamServicePort.QueryFrame(),
                delegate(webcam.QueryFrameResponse success)
                {
                    response = success;
                },
                delegate(Fault f)
                {
                    fault = f;
                }
            );

            if (fault != null)
            {
                yield break;
            }


            if (grayImgOld == null)
            {
                grayImg = new byte[response.Size.Height * response.Size.Width];
                grayImgOld = new byte[response.Size.Height * response.Size.Width];
                motionImg = new byte[response.Size.Height * response.Size.Width];
                _state.MotionFrame = new byte[response.Size.Height * response.Size.Width];
                _state.MotionFrameWidth = response.Size.Width;
                _state.MotionFrameHeight = response.Size.Height;
                yield break;
            }


            for (int i = 0, j = 0; i < grayImg.Length; i++, j += 3)
            {
                grayImg[i] = (byte)(((int)response.Frame[j] + (int)response.Frame[j + 1] + (int)response.Frame[j + 2]) / 3);
            }

            // calculate frame difference
            for (int i = 0; i < response.Size.Height * response.Size.Width; i++)
            {
                motionImg[i] = (byte)Math.Abs(grayImg[i] - grayImgOld[i]);
            }

            
            bool motionFound = DetectMotionAndUpdateState(motionImg, response.Size.Width, response.Size.Height, 50, 1500);
            
            if (motionFound)
            {
                //LogInfo("Motion Found!");
                base.SendNotification(_submgrPort, new MotionFrameAvailable());
            }
            motionImg.CopyTo(_state.MotionFrame, 0);
            grayImg.CopyTo(grayImgOld, 0);

        }

        [ServiceHandler(ServiceHandlerBehavior.Exclusive)]
        public void DisableHandler(Disable dsl)
        {
            _state.Enabled = false;
        }

        [ServiceHandler(ServiceHandlerBehavior.Exclusive)]
        public void EnableHandler(Enable dsl)
        {
            _state.Enabled = true;
        }

    

        /// <summary>
        /// Handles Subscribe messages
        /// </summary>
        /// <param name="subscribe">the subscribe request</param>
        [ServiceHandler]
        public void SubscribeHandler(Subscribe subscribe)
        {
            SubscribeHelper(_submgrPort, subscribe.Body, subscribe.ResponsePort);
        }



        bool DetectMotionAndUpdateState(byte[] image, int imageWidth, int imageHeight, byte diffThresh, int sizeThresh)
        {
            int i, x, y;

            for (i = 0; i < imageWidth * imageHeight; i++)
            {
                if (image[i] > diffThresh)
                    image[i] = 255;
                else
                    image[i] = 0;
            }

            int sumX = 0;
            int sumY = 0;
            int pixelCount = 0;
            for (y = 0; y < imageHeight; y++)
            {
                for (x = 0; x < imageWidth; x++)
                {
                    if (image[y * imageWidth + x] != 0)
                    {
                        sumX += x;
                        sumY += y;
                        pixelCount++;
                    }
                }
            }

            if (pixelCount > sizeThresh)
            {
                _state.MotionDetected = true;
                _state.XMotion = (sumX / pixelCount);
                _state.YMotion = (sumY / pixelCount);
                _state.MotionSize = pixelCount;
            }
            else
                _state.MotionDetected = false;

            return _state.MotionDetected;
        }


    }
}


