using System;
using System.Collections.Generic;
using System.ComponentModel;
using Microsoft.Ccr.Core;
using Microsoft.Dss.Core.Attributes;
using Microsoft.Dss.ServiceModel.Dssp;
using Microsoft.Dss.ServiceModel.DsspServiceBase;
using W3C.Soap;
using submgr = Microsoft.Dss.Services.SubscriptionManager;
using System.Runtime.InteropServices;

namespace PanTiltCamControl
{
    [Contract(Contract.Identifier)]
    [DisplayName("PanTiltCamControl")]
    [Description("PanTiltCamControl service (no description provided)")]
    class PanTiltCamControlService : DsspServiceBase
    {
        /// <summary>
        /// Service state
        /// </summary>
        [ServiceState]
        PanTiltCamControlState _state = new PanTiltCamControlState();

        /// <summary>
        /// Main service port
        /// </summary>
        [ServicePort("/PanTiltCamControl", AllowMultipleInstances = true)]
        PanTiltCamControlOperations _mainPort = new PanTiltCamControlOperations();

        [SubscriptionManagerPartner]
        submgr.SubscriptionManagerPort _submgrPort = new submgr.SubscriptionManagerPort();

        /// <summary>
        /// Service constructor
        /// </summary>
        public PanTiltCamControlService(DsspServiceCreationPort creationPort)
            : base(creationPort)
        {
        }

        /// <summary>
        /// Service start
        /// </summary>
        protected override void Start()
        {

            int NumCams = GetDeviceCount();
            LogInfo("Nombre de camras: " + NumCams);

            // Look for a camera 
            IntPtr path = IntPtr.Zero;
            for (int i = 0; i < NumCams; i++)
            {
                path = GetDevicePath(i);
                LogInfo("Camera trouve: " + path.ToString());
            }

            // Open the right device here
            path = GetDevicePath(1);
            _state.CameraHandle = OpenDevice(path);

            if (_state.CameraHandle != 0)
            {
                LogInfo("Utilisation de la camra: " + _state.CameraHandle);
            }
            else
            {
                LogError("Pas de camra, pas de chocolat!");
            }

            base.Start();
        }

        /// <summary>
        /// Handles Subscribe messages
        /// </summary>
        /// <param name="subscribe">the subscribe request</param>
        [ServiceHandler]
        public void SubscribeHandler(Subscribe subscribe)
        {
            SubscribeHelper(_submgrPort, subscribe.Body, subscribe.ResponsePort);
        }


        [ServiceHandler(ServiceHandlerBehavior.Exclusive)]
        public void PanTiltHandler(PanTilt request)
        { 
            switch(request.Body.Move)
            {
                case PanTiltMovement.PanLeft:
                    LogInfo("Requested Move: Pan Left");
                    MoveLeft(_state.CameraHandle);
                    break;
                case PanTiltMovement.PanRight:
                    LogInfo("Requested Move: Pan Right");
                    MoveRight(_state.CameraHandle);
                    break;
                case PanTiltMovement.TiltDown:
                    LogInfo("Requested Move: Tilt Down");
                    MoveBottom(_state.CameraHandle);
                    break;
                case PanTiltMovement.TiltUp:
                    LogInfo("Requested Move: Tilt Up");
                    MoveTop(_state.CameraHandle);
                    break;
                case PanTiltMovement.ResetPOV:
                    LogInfo("Requested Move: Reset POV");
                    MoveHome(_state.CameraHandle);
                    break;
                default:
                    LogError("I don't know this move");
                    break;
            }

            request.ResponsePort.Post(DefaultUpdateResponseType.Instance);

        }


        // Load Logitech driver operations
        [DllImport("logimove.dll")]
        static extern void ReloadDevices();
        //Reloads the list of system devices to operate on.

        [DllImport("logimove.dll")]
        static extern int GetDeviceCount();
        //Retrieves the number of system devices to operate on.

        [DllImport("logimove.dll")]
        static extern IntPtr GetDevicePath(int iDeviceIndex);
        //Retrieves the path of a device. iDeviceIndex is in range of 0 to GetDeviceCount()-1.

        [DllImport("logimove.dll")]
        static extern int OpenDevice(IntPtr DevicePath);
        //Opens the device with specified path, and returns a handle to it.
        //If failed, returns 0.
        //Note: HANDLE is a 32-bit integer.

        [DllImport("logimove.dll")]
        static extern int CloseDevice(int hDevice);
        //Closes an opened device, by its handle. Returns whether operation succeeded or not.

        [DllImport("logimove.dll")]
        static extern void MoveLeft(int hDevice);
        //Pans camera, specified by device handle, to the left.

        [DllImport("logimove.dll")]
        static extern void MoveRight(int hDevice);
        //Pans camera, specified by device handle, to the right.

        [DllImport("logimove.dll")]
        static extern void MoveTop(int hDevice);
        //Tilts camera, specified by device handle, up.

        [DllImport("logimove.dll")]
        static extern void MoveBottom(int hDevice);
        //Tilts camera, specified by device handle, down.

        [DllImport("logimove.dll")]
        static extern void MoveHome(int hDevice);
        //Moves the camera to its home position.

        [DllImport("logimove.dll")]
        static extern void Move(int hDevice, int iPosition);
        //* MOVE_HOME = 0x00000000
        //* MOVE_LEFT = 0x0000FF80
        //* MOVE_RIGHT = 0x00000080
        //* MOVE_TOP = 0xFF800000
        //* MOVE_BOTTOM = 0x00800000



    }
}


