using System;
using System.Collections.Generic;
using System.ComponentModel;
using Microsoft.Ccr.Core;
using Microsoft.Dss.Core.Attributes;
using Microsoft.Dss.ServiceModel.Dssp;
using Microsoft.Dss.ServiceModel.DsspServiceBase;
using W3C.Soap;
using submgr = Microsoft.Dss.Services.SubscriptionManager;
using ptcc = PanTiltCamControl.Proxy;
using webcam = Microsoft.Robotics.Services.WebCam.Proxy;


using ccrwpf = Microsoft.Ccr.Adapters.Wpf;
using System.Windows;
using System.Drawing.Imaging;
using System.Drawing;
using System.IO;
using System.Windows.Media.Imaging;

using motion = MotionDetector.Proxy;
using System.Windows.Media;


/*************************************************************************************************************
 * ATTENTION IMPORTANT
 * Cette dtection de mouvement est une base plus qu'imparfaite
 * Les algorithmes de dtection de mouvement pros sont beaucoup plus compliqus
 * Smoothing de l'image, stitching des frames et oprations de morphologie sont ncessaires pour liminer 
 * le "bruit" li au mouvement de la camra en elle-mme
 * il faudrait ensuite adapter l'amplitude du mouvement de la camra proportionnellement  la position du centre
 * du mouvement dtect.
 * 
 * Ce service n'a pour but que de montrer comment insrer un filtre dans une architecture MSRDS
 *************************************************************************************************************/



namespace PanTiltCamGUI
{
    [Contract(Contract.Identifier)]
    [DisplayName("PanTiltCamGUI")]
    [Description("PanTiltCamGUI service (no description provided)")]
    public class PanTiltCamGUIService : DsspServiceBase
    {
        /// <summary>
        /// Service state
        /// </summary>
        [ServiceState]
        PanTiltCamGUIState _state = new PanTiltCamGUIState();

        /// <summary>
        /// Main service port
        /// </summary>
        [ServicePort("/PanTiltCamGUI", AllowMultipleInstances = true)]
        PanTiltCamGUIOperations _mainPort = new PanTiltCamGUIOperations();

        [SubscriptionManagerPartner]
        submgr.SubscriptionManagerPort _submgrPort = new submgr.SubscriptionManagerPort();

        /// <summary>
        /// PanTiltCamControlService partner
        /// </summary>
        [Partner("PanTiltCamControlService", Contract = ptcc.Contract.Identifier, CreationPolicy = PartnerCreationPolicy.UseExistingOrCreate)]
        ptcc.PanTiltCamControlOperations _panTiltCamControlServicePort = new ptcc.PanTiltCamControlOperations();
        ptcc.PanTiltCamControlOperations _panTiltCamControlServiceNotify = new ptcc.PanTiltCamControlOperations();

        /// <summary>
        /// WebCamService partner
        /// </summary>
        [Partner("/WebcamService", Contract = webcam.Contract.Identifier, CreationPolicy = PartnerCreationPolicy.UseExistingOrCreate)]
        webcam.WebCamOperations _webcamServicePort = new Microsoft.Robotics.Services.WebCam.Proxy.WebCamOperations();
        webcam.WebCamOperations _webcamServiceNotify = new Microsoft.Robotics.Services.WebCam.Proxy.WebCamOperations();


        [Partner("/MotionDetector", Contract = motion.Contract.Identifier, CreationPolicy = PartnerCreationPolicy.UseExistingOrCreate)]
        motion.MotionDetectorOperations _motionDetectorPort = new motion.MotionDetectorOperations();
        motion.MotionDetectorOperations _motionDetectorNotify = new motion.MotionDetectorOperations();


        ccrwpf.WpfServicePort _wpfServicePort;
        PanTiltCamGUI _userInterface;

        
        
        /// <summary>
        /// Service constructor
        /// </summary>
        public PanTiltCamGUIService(DsspServiceCreationPort creationPort)
            : base(creationPort)
        {
        }

        /// <summary>
        /// Service start
        /// </summary>
        protected override void Start()
        {

            // 
            // Add service specific initialization here
            // 
            SpawnIterator(Initialize);
            
        }

        IEnumerator<ITask> Initialize()
        {
            
            #region Create UI
            _wpfServicePort = ccrwpf.WpfAdapter.Create(TaskQueue);
            var runWindow = _wpfServicePort.RunWindow(() => new PanTiltCamGUI(this));
            yield return (Choice)runWindow;

            var exception = (Exception)runWindow;
            if (exception != null)
            {
                LogError(exception);
                StartFailed();
                yield break;
            }

            _userInterface = (Window)runWindow as PanTiltCamGUI;

            #endregion

            #region Subscribe to webcam notifications
            var subscribe = _webcamServicePort.Subscribe(_webcamServiceNotify, typeof(webcam.UpdateFrame));
            yield return (Choice)subscribe;

            var fault = (Fault)subscribe;
            if (fault != null)
            {
                LogError(fault);
                StartFailed();
                yield break;
            }
            #endregion

            #region Subscribe to motion detection notifications
            subscribe = _motionDetectorPort.Subscribe(_motionDetectorNotify, typeof(motion.MotionFrameAvailable));
            yield return (Choice)subscribe;

            fault = (Fault)subscribe;
            if (fault != null)
            {
                LogError(fault);
                StartFailed();
                yield break;
            }
            #endregion


            StartComplete();
        }

        void StartComplete()
        {
            base.Start();

            MainPortInterleave.CombineWith(
                Arbiter.Interleave(
                    new TeardownReceiverGroup(),
                    new ExclusiveReceiverGroup(
                        Arbiter.Receive<webcam.UpdateFrame>(true, _webcamServiceNotify, UpdateFrameHandler),
                        Arbiter.Receive<motion.MotionFrameAvailable>(true, _motionDetectorNotify, MotionFrameAvailableHandler)
                        ),
                    new ConcurrentReceiverGroup()
                )
            );
        }


        /// <summary>
        /// Handles Subscribe messages
        /// </summary>
        /// <param name="subscribe">the subscribe request</param>
        [ServiceHandler]
        public void SubscribeHandler(Subscribe subscribe)
        {
            SubscribeHelper(_submgrPort, subscribe.Body, subscribe.ResponsePort);
        }

        public void MotionFrameAvailableHandler(motion.MotionFrameAvailable motionFrameAvailableNotification)
        {
            SpawnIterator(MotionFrameAvailableHandlerIterator);
        }

        public IEnumerator<ITask> MotionFrameAvailableHandlerIterator()
        {
            Fault fault = null;
            motion.MotionDetectorState mds = null;

            yield return Arbiter.Choice(
                _motionDetectorPort.Get(),
                delegate(motion.MotionDetectorState success)
                {
                    mds = success;
                },
                delegate(Fault f)
                {
                    fault = f;
                });

            if (fault != null)
            {
                yield break;
            }

            if (mds.XMotion > 2 * mds.MotionFrameWidth / 3)
            {
                yield return Arbiter.Choice(
                    _panTiltCamControlServicePort.PanTilt(new ptcc.PanTiltRequest() { Move = ptcc.PanTiltMovement.PanRight }),
                    success => { },
                    failure => fault = failure
                    );

            }
            else if (mds.XMotion < mds.MotionFrameWidth / 3)
            {
                yield return Arbiter.Choice(
                    _panTiltCamControlServicePort.PanTilt(new ptcc.PanTiltRequest() { Move = ptcc.PanTiltMovement.PanLeft }),
                    success => { },
                    failure => fault = failure
                    );
            }

            if (mds.YMotion > 2 * mds.MotionFrameHeight / 3)
            {
                yield return Arbiter.Choice(
                    _panTiltCamControlServicePort.PanTilt(new ptcc.PanTiltRequest() { Move = ptcc.PanTiltMovement.TiltUp }),
                    success => { },
                    failure => fault = failure
                    );

            }
            else if (mds.YMotion < mds.MotionFrameHeight / 3)
            {
                yield return Arbiter.Choice(
                    _panTiltCamControlServicePort.PanTilt(new ptcc.PanTiltRequest() { Move = ptcc.PanTiltMovement.TiltDown }),
                    success => { },
                    failure => fault = failure
                    );

            }

            _wpfServicePort.Invoke(() =>
                {
                    
                    _userInterface.imageMotion.Source = BitmapFrame.Create(mds.MotionFrameWidth, mds.MotionFrameHeight, 96.0, 96.0,
                                                                           PixelFormats.Gray8, BitmapPalettes.WebPalette,
                                                                           mds.MotionFrame, mds.MotionFrameWidth);
                });

        }

        public void UpdateFrameHandler(webcam.UpdateFrame update)
        {
            SpawnIterator(update, UpdateFrameHandlerIterator);
        }


        public IEnumerator<ITask> UpdateFrameHandlerIterator(webcam.UpdateFrame update)
        {
            
            Fault fault = null;
            webcam.QueryFrameRequest request = new webcam.QueryFrameRequest();
            webcam.QueryFrameResponse response = null;

            request.Format = ImageFormat.Jpeg.Guid;
            request.Size = new Microsoft.Robotics.PhysicalModel.Proxy.Vector2(640, 480);

            yield return Arbiter.Choice(
                _webcamServicePort.QueryFrame(request),
                delegate(webcam.QueryFrameResponse success)
                {
                    response = success;
                },
                delegate(Fault f)
                {
                    fault = f;
                }
            );

            if (fault != null)
            {
                yield break;
            }


            _state.LastFrameStream = new MemoryStream(response.Frame, false);
            _wpfServicePort.Invoke(() =>
            {
                try
                {
                    _state.LastFrameStream.Seek(0, SeekOrigin.Begin);
                    _userInterface.image1.Source = BitmapFrame.Create(_state.LastFrameStream, BitmapCreateOptions.None, BitmapCacheOption.OnLoad);
                    
                }
                catch (Exception ex)
                { 
                    LogError(ex.Message);
                }
                });
        
        }


        internal void Button_Click(string cmd)
        {
            ptcc.PanTiltMovement RequestedMove;
            switch (cmd)
            {
                case "up":
                    RequestedMove = ptcc.PanTiltMovement.TiltUp;
                    break;
                case "down":
                    RequestedMove = ptcc.PanTiltMovement.TiltDown;
                    break;
                case "left":
                    RequestedMove = ptcc.PanTiltMovement.PanLeft;
                    break;
                case "right":
                    RequestedMove = ptcc.PanTiltMovement.PanRight;
                    break;
                case "reset":
                    RequestedMove = ptcc.PanTiltMovement.ResetPOV;
                    break;
                
                default:
                    RequestedMove = ptcc.PanTiltMovement.ResetPOV;
                    break;
            }

            Activate(
                Arbiter.Choice(
                    _panTiltCamControlServicePort.PanTilt(new ptcc.PanTiltRequest() { Move = RequestedMove } ),
                    success => { /* Do nothing */ },
                    fault =>
                    {
                        _wpfServicePort.Invoke(() => _userInterface.ShowFault(fault));
                    }
                )
            );
        }

        internal void ToggleMotionDetection(string cmd)
        {
            switch (cmd)
            {
                case "EnableMotionDetection":
                    Activate(
                        Arbiter.Choice(
                        _motionDetectorPort.Enable(),
                        success => { },
                        fault =>
                        {
                            _wpfServicePort.Invoke(() => _userInterface.ShowFault(fault));
                        }
                    ));
                    break;
                case "DisableMotionDetection":
                    Activate(
                        Arbiter.Choice(
                        _motionDetectorPort.Disable(),
                        success => { },
                        fault =>
                        {
                            _wpfServicePort.Invoke(() => _userInterface.ShowFault(fault));
                        }
                    ));
                    break;
                default:
                    break;
            }
        }

    }
}


