﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Navigation;
using System.Windows.Shapes;

namespace PieMenuSample
{
    [TemplatePart(Name = "PART_PieContent", Type = typeof(Grid))]
    [TemplatePart(Name = "PART_PieBorderPath", Type = typeof(Path))]
    [TemplatePart(Name = "PART_ClippedPanel", Type = typeof(Panel))]
    public class PieContentControl : ContentControl
    {
        #region Definition des membres de classe
        // Attributs privés qui serviront à stocker les références sur les éléments
        // significatifs du template courant
        private Grid        _PartPieContent     = null;
        private Path        _PartPieBorderPath  = null;
        private Panel       _PartClippedPanel   = null;

        // Geometry représentant la forme de "Pie" prise par le PieContentControl
        private Geometry    _PieGeometry = null;

        #endregion

        #region Définition des assesseurs
        // La propriété MinAngle définie, en degré, l'angle minimal du Pie
        public Double MinAngle
        {
            get { return (Double)GetValue(MinAngleProperty); }
            set { SetValue(MinAngleProperty, value); }
        }

        // La propriété MaxAngle définie, en degré, l'angle maximal du Pie
        public Double MaxAngle
        {
            get { return (Double)GetValue(MaxAngleProperty); }
            set { SetValue(MaxAngleProperty, value); }
        }

        // Cette propriété, entre 0 et 1, définie la distance entre le contenu du PieContentControl
        // et les extremités de celui ci : Si 0, le contenu est au "centre" du cercle. Si 1, il
        // est en bordure de celui ci.
        public Double ContentDistance
        {
            get { return (Double)GetValue(ContentDistanceProperty); }
            set { SetValue(ContentDistanceProperty, value); }
        }

        // Translation radiale exercée sur le Pie, selon son angle médian.
        public Double RadialOffset
        {
            get { return (Double)GetValue(RadialOffsetProperty); }
            set { SetValue(RadialOffsetProperty, value); }
        }
        #endregion

        #region Définition des propriétés de dépendances
        public static readonly DependencyProperty MinAngleProperty;
        public static readonly DependencyProperty MaxAngleProperty;
        public static readonly DependencyProperty ContentDistanceProperty;
        public static readonly DependencyProperty RadialOffsetProperty;
        #endregion

        // Le constructeur statique sert à initialiser les propriétés de dépendances.
        static PieContentControl()
        {
            DefaultStyleKeyProperty.OverrideMetadata(typeof(PieContentControl), new FrameworkPropertyMetadata(typeof(PieContentControl)));

            // Initialisation de chacune des propriétés de dépendances en précisant :
            // - Une valeur par défaut
            // - Le flag précisant que l'organisation du contenu devra être remis à jour lors de la modification de la propriété ( AffectsArrange ) : ArrangeOverride sera alors réinvoquée
            MinAngleProperty = DependencyProperty.Register("MinAngle", typeof(Double), typeof(PieContentControl), new FrameworkPropertyMetadata(0.0, FrameworkPropertyMetadataOptions.AffectsArrange));
            MaxAngleProperty = DependencyProperty.Register("MaxAngle", typeof(Double), typeof(PieContentControl), new FrameworkPropertyMetadata(359.99, FrameworkPropertyMetadataOptions.AffectsArrange));
            ContentDistanceProperty = DependencyProperty.Register("ContentDistance", typeof(Double), typeof(PieContentControl), new FrameworkPropertyMetadata(0.5, FrameworkPropertyMetadataOptions.AffectsArrange));
            RadialOffsetProperty = DependencyProperty.Register("RadialOffset", typeof(Double), typeof(PieContentControl), new FrameworkPropertyMetadata(0.0, FrameworkPropertyMetadataOptions.AffectsArrange));
        }

        public PieContentControl()
        {
        }

        private Geometry computePieGeometry(Point ptCenter, double dMinRadiusX, double dMinRadiusY, double dMaxRadiusX, double dMaxRadiusY, double dMinAngle, double dMaxAngle)
        {
            // Le calcul de notre geometry se base sur des manipulations de l'ellipse exterieure de celle ci.
            Geometry geoPieShape = new EllipseGeometry(ptCenter, dMaxRadiusX, dMaxRadiusY) as Geometry;

            // On evide eventuellement l'ellipse de son centre.
            if ((dMinRadiusX != 0) || (dMinRadiusY!=0 ))
            {
                EllipseGeometry elpMinBound = new EllipseGeometry(ptCenter, dMinRadiusX, dMinRadiusY);
                geoPieShape = Geometry.Combine ( geoPieShape, elpMinBound, GeometryCombineMode.Exclude, null );
            }

            double dMaxRadius = Math.Max(dMaxRadiusX, dMaxRadiusY);

            // Pour déterminer le secteur angulaire correspondant, on va travailler par combinaisons avec
            // les deux rectangles complémentaires ici définis. Ils serviront de masques pour n'afficher
            // que le secteur désiré

            // Si l'angle du secteur angulaire est < 180°, alors ces deux rectangles, sur lesquels les
            // rotations correspondants aux angles Min et Max du Secteur angulaire, seront retranchés de la
            // géometry actuelle : On obtiendra ainsi la forme souhaitée.

            // Si l'angle du secteur angulaire est > 180°, alors on ne gardera, de la Geometry actuelle, 
            // que la partie superposant les deux rectangles (partie complémentaire à la précédente)
            RectangleGeometry rctMinBound = new RectangleGeometry(new Rect( ptCenter.X - dMaxRadius - 5,
                                                                            ptCenter.Y - dMaxRadius - 5,
                                                                            2.0 * dMaxRadius + 10,
                                                                            dMaxRadius + 5));

            RectangleGeometry rctMaxBound = new RectangleGeometry(new Rect( ptCenter.X - dMaxRadius - 5,
                                                                            ptCenter.Y,
                                                                            2.0 * dMaxRadius + 10,
                                                                            dMaxRadius + 5));
            // On fait effectuer une rotation à nos "deux rectangles pochoirs"
            rctMinBound.Transform = new RotateTransform(dMinAngle, ptCenter.X, ptCenter.Y);
            rctMaxBound.Transform = new RotateTransform(dMaxAngle, ptCenter.X, ptCenter.Y);

            // .. Et on les combine logiquement pour arriver à la forme désirée.
            if (Math.Abs((dMaxAngle - dMinAngle) % 360) <= 180)
            {
                geoPieShape = Geometry.Combine(geoPieShape, rctMinBound, GeometryCombineMode.Exclude, null);
                geoPieShape = Geometry.Combine(geoPieShape, rctMaxBound, GeometryCombineMode.Exclude, null);
            }
            else
            {
                Geometry geoFirstShape = Geometry.Combine(geoPieShape, rctMinBound, GeometryCombineMode.Exclude, null);
                Geometry geoSecondShape = Geometry.Combine(geoPieShape, rctMaxBound, GeometryCombineMode.Exclude, null);
                geoPieShape = Geometry.Combine(geoFirstShape, geoSecondShape, GeometryCombineMode.Union, null);
            }

            // Il y a évidemment d'autres moyens d'arriver à ce résultat. Voyez par exemple ici :
            // http://lamp.codeproject.com/KB/WPF/PieChartDataBinding.aspx?msg=2682654#xx2682654xx

            return (geoPieShape);
        }

        private void updatePie(Size availableSize)
        {
            // On détermine le centre du Pie, les rayons X et Y, et l'angle median
            Point ptCenter = new Point(availableSize.Width / 2.0, availableSize.Height / 2.0);
            double dRadiusX = availableSize.Width / 2.0;
            double dRadiusY = availableSize.Height / 2.0;
            double dMidAngle = (MinAngle + MaxAngle) / 2.0 * Math.PI / 180.0;

            // Construction de la géométrie du Pie. Il s'agit d'un quartier de l'ellipse "théorique" d'origine,
            // de rayon dRadiusX et dRadiusY, que l'on va trancher entre les angles MinAngle et MaxAngle.
            _PieGeometry = computePieGeometry(ptCenter, 0, 0, dRadiusX, dRadiusY, MinAngle, MaxAngle);

            // En cas d'echec de la construction, on s'arrete la...
            if (_PieGeometry.IsEmpty())
            {
                return;
            }

            // On fixe la geometry ainsi calculée comme nouvelle zone de clipping de notre objet graphique.
            // En faisant cela, on ne gardera, du fond graphique réalisé par le designer, que le graphisme 
            // correspondant à notre tranche
            if (_PartClippedPanel != null)
            {
                _PartClippedPanel.Clip = _PieGeometry;
            }

            // On fixe aussi la géométrie précédemment calculée comme géométrie du Path de Bordure, s'il 
            // est défini. Cela permettra de "fermer" la zone de clipping et de finaliser le look graphique de notre Pie. 
            if (_PartPieBorderPath != null)
            {
                _PartPieBorderPath.Data = _PieGeometry;
                _PartPieBorderPath.Margin = new Thickness ( _PieGeometry.Bounds.Left,
                                                            _PieGeometry.Bounds.Top, 
                                                            0, 
                                                            0 );
                _PartPieBorderPath.Width = _PieGeometry.Bounds.Width;
                _PartPieBorderPath.Height = _PieGeometry.Bounds.Height;
            }
    
            // On position le contenu du PieContentControl. Lors du changement de template, on le fixe
            // au centre de notre PieContentControl. Ici, on va simplement le translater pourqu'il soit
            // au milieu de la tranche découpée.
            if (_PartPieContent != null)
            {
                // L'élément est situé à un facteur ContentDistance du bord de l'ellipse...
                double dOffsetX = Math.Cos(dMidAngle) * dRadiusX * ContentDistance;
                double dOffsetY = Math.Sin(dMidAngle) * dRadiusY * ContentDistance;
                // .. et on applique la Translation correspondante.
                _PartPieContent.RenderTransform = new TranslateTransform(dOffsetX, dOffsetY);
            }


            // Si un offset radial est défini, on translate la tranche selon l'angle médian
            if (RadialOffset != 0.0)
            {
                double dRadialOffsetX = Math.Cos(dMidAngle) * RadialOffset;
                double dRadialOffsetY = Math.Sin(dMidAngle) * RadialOffset;
                this.RenderTransform = new TranslateTransform(dRadialOffsetX, dRadialOffsetY);
            }
        }
        protected override Size ArrangeOverride(Size arrangeBounds)
        {
            // On invoque la méthode d'organisation de base, et on récupère la taille finale..
            Size finalSize = base.ArrangeOverride(arrangeBounds);
            // .. et on met à jour notre Pie en fonction de cette taille
            updatePie(finalSize);
            return (finalSize);
        }

        public override void OnApplyTemplate()
        {
            base.OnApplyTemplate();

            // Récupération du template part "PART_PieContent"...
            _PartPieContent = Template.FindName("PART_PieContent", this) as Grid;
            // .. que l'on centre, par défaut. Cela nous permettra d'avoir un controle fin
            // sur sa position, pour le fixer aisément par rapport au centre de l'ellipse théorique.
            if (_PartPieContent != null)
            {
                _PartPieContent.HorizontalAlignment = HorizontalAlignment.Center;
                _PartPieContent.VerticalAlignment = VerticalAlignment.Center;
            }
            // Récupération du template part "PART_PieBorderPath", qui précise le path qui sera utilisé
            // pour le bord de la tranche
            _PartPieBorderPath = Template.FindName("PART_PieBorderPath", this) as Path;
            // S'il existe, on le force en haut à gauche
            if (_PartPieBorderPath != null)
            {
                _PartPieBorderPath.HorizontalAlignment = HorizontalAlignment.Left;
                _PartPieBorderPath.VerticalAlignment = VerticalAlignment.Top;
            }
            // Récupération du panel qui sera clippé par notre géometry de tranche.
            _PartClippedPanel = Template.FindName("PART_ClippedPanel", this) as Panel;
        }
    }
}
