﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Media;
using System.ComponentModel;

namespace PieMenuSample
{
    public class RadialPanel : Panel
    {
        #region Définition des assesseurs sur les propriétés attachées
        
        // AngleMin est une propriété attachée que fixera le RadialPanel sur ses fils.
        // Ceux ci devront ajuster leur affichage en conséquence.
        public static Double GetAngleMin(DependencyObject obj)
        {
            return (Double)obj.GetValue(AngleMinProperty);
        }
        
        // AngleMax est une propriété attachée que fixera le RadialPanel sur ses fils.
        // Ceux ci devront ajuster leur affichage en conséquence.
        public static Double GetAngleMax(DependencyObject obj)
        {
            return (Double)obj.GetValue(AngleMaxProperty);
        }
        #endregion

        #region Définition des propriétés attachées

        // Les propriétés attachées AngleMinKey et AngleMaxKey sont définies en lecture seules :
        // Elles sont donc rattachées à une clef qui permet de fixer leur valeur, pour quiconque possède
        // une visibilité dessus. Les clefs sont définies comme privées pour que seul RadialPanel y ait accès.
        private static readonly DependencyPropertyKey AngleMinKey;
        private static readonly DependencyPropertyKey AngleMaxKey;

        public static readonly DependencyProperty AngleMinProperty;
        public static readonly DependencyProperty AngleMaxProperty;
        #endregion

        static RadialPanel()
        {
            // On enregistre les propriétés en lecture seule. La clef permettra de fixer leur valeur.
            AngleMinKey = DependencyProperty.RegisterAttachedReadOnly("AngleMin", typeof(Double), typeof(RadialPanel), new FrameworkPropertyMetadata(0.0, FrameworkPropertyMetadataOptions.AffectsRender));
            AngleMaxKey = DependencyProperty.RegisterAttachedReadOnly("AngleMax", typeof(Double), typeof(RadialPanel), new FrameworkPropertyMetadata(359.0, FrameworkPropertyMetadataOptions.AffectsRender));
            AngleMinProperty = AngleMinKey.DependencyProperty;
            AngleMaxProperty = AngleMaxKey.DependencyProperty;
        
        }

        public RadialPanel()
        {
        }

        protected override Size MeasureOverride(Size availableSize)
        {
            Size resultSize = new Size();
            
            // MeasureOverride est une méthode qui permet à un layout de notifier sa taille idéale.
            // Ici, on se contente de renvoyer la taille disponible pour le layout.
            // Si celle ci est infinie, on renvoit la taille du plus grand de ses fils.
            foreach (UIElement eltChild in this.Children)
            {
                eltChild.Measure(availableSize);
                resultSize.Width = Math.Max(resultSize.Width, eltChild.DesiredSize.Width);
                resultSize.Height = Math.Max(resultSize.Height, eltChild.DesiredSize.Height);
            }

            resultSize.Width = double.IsPositiveInfinity(availableSize.Width) ? resultSize.Width : availableSize.Width;
            resultSize.Height = double.IsPositiveInfinity(availableSize.Height) ? resultSize.Height : availableSize.Height;

            return resultSize;
        }

        protected override Size ArrangeOverride(Size finalSize)
        {
            if (Children.Count != 0)
            {
                // Position du centre du panneau circulaire
                Point ptCentre = new Point(finalSize.Width / 2.0, finalSize.Height / 2.0);

                // Pas angulaire
                double dAngularPace = (double)(360.0 / Children.Count);
                // Angle courant
                double dCurrentAngle = 0.0;

                // ArrangeOverride sert à positionner les fils d'un layout sur sa surface. 
                // Dans notre cas, on étend les fils sur toute la surface disponible, et on en
                // profite pour fixer le secteur angulaire correspondant.
                foreach (UIElement eltChild in Children)
                {
                    eltChild.Arrange(new Rect(0,
                                              0,
                                              finalSize.Width,
                                              finalSize.Height));

                    eltChild.SetValue(AngleMinKey, dCurrentAngle);
                    eltChild.SetValue(AngleMaxKey, dCurrentAngle + dAngularPace);

                    dCurrentAngle += dAngularPace;
                }

            }
            return finalSize;
        }
    }
}
