using System;
using System.Collections.Generic;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Audio;
using Microsoft.Xna.Framework.Graphics;
using Microsoft.Xna.Framework.Input;
using Microsoft.Xna.Framework.Storage;
using Microsoft.Xna.Framework.Content;

using System.Diagnostics;
namespace XnaPanic
{
    public partial class Monster
    {
        #region Constants
        private const int UpdatePeriod = 75;
        private const int ExitCycle = 50;
        #endregion

        #region Static fields
        // Shared sprites between all Monsters
        private static MultiTextureSprite[] _sprites;
        private static MultiTextureSprite[] _inHoleSprites;
        // Random generator
        static Random rnd = new Random();
        #endregion

        #region Instance Fields
        private int _level;
        private MultiTextureSprite _sprite;
        private Grid _grid;
        private BoundingBox _bb;
        private int _index;
        private bool _inHole;
        private GameScreen _gameScreen;
        private ISoundService _soundService;

        int _x;
        int _y;
        int _spriteX;
        int _spriteY;
        int _oldY;
        int _speedX;
        int _speedY;
        int _oldSpeedX;
        bool _falling;
        int _startFallY;
        bool _killed;
        int _height;
        bool _changeDir;
        Cue _cue;
        #endregion

        #region Properties
        public int Index
        {
            get { return _index; }
            set { _index = value; }
        }
        public bool ChangeDir
        {
            get { return _changeDir; }
            set { _changeDir = value; }
        }
        public bool Dead
        {
            get
            {
                return _killed;
            }
        }
        public bool Falling
        {
            get
            {
                return _falling;
            }
        }
        public BoundingBox BoundingBox
        {
            get
            {
                return _bb;
            }
        }
        public bool InHole
        {
            get
            {
                return _inHole;
            }
        }
        public int X
        {
            get { return _x; }
            set { _x = value; }
        }
        public int Y
        {
            get { return _y; }
            set { _y = value; }
        }
        #endregion

        #region Construction
        public Monster(int level, GameScreen gameScreen, Grid grid, Game game)
        {
            _gameScreen = gameScreen;
            _grid = grid;
            _level = level;
            _speedX = 0;
            _speedY = 0;
            _inHole = false;
            _falling = false;
            _killed = false;
            _sprite = _sprites[_level - 1];
            _changeDir = false;
            _soundService = (ISoundService)game.Services.GetService(typeof(ISoundService));
        }
        #endregion


        static public void LoadGraphicsContent(ContentManager content, int width, int height) {
            _sprites = new MultiTextureSprite[3];
            _sprites[0] = new MultiTextureSprite(SpriteHorizontalAlignment.Center, SpriteVerticalAlignment.Bottom, content, @"Sprites\Monsters", "Red1", "Red2");
            _sprites[0].OffsetY = 0;
            _sprites[1] = new MultiTextureSprite(SpriteHorizontalAlignment.Center, SpriteVerticalAlignment.Bottom, content, @"Sprites\Monsters", "Green1", "Green2");
            _sprites[1].OffsetY = 0;
            _sprites[2] = new MultiTextureSprite(SpriteHorizontalAlignment.Center, SpriteVerticalAlignment.Bottom, content, @"Sprites\Monsters", "White1", "White2");
            _sprites[2].OffsetY = 0;

            _inHoleSprites = new MultiTextureSprite[3];
            _inHoleSprites[0] = new MultiTextureSprite(SpriteHorizontalAlignment.Center, SpriteVerticalAlignment.Top, content, @"Sprites\Monsters", "RedHole1", "RedHole2");
            _inHoleSprites[0].OffsetY = -36;
            _inHoleSprites[1] = new MultiTextureSprite(SpriteHorizontalAlignment.Center, SpriteVerticalAlignment.Top, content, @"Sprites\Monsters", "GreenHole1", "GreenHole2");
            _inHoleSprites[1].OffsetY = -36;
            _inHoleSprites[2] = new MultiTextureSprite(SpriteHorizontalAlignment.Center, SpriteVerticalAlignment.Top, content, @"Sprites\Monsters", "WhiteHole1", "WhiteHole2");
            _inHoleSprites[2].OffsetY = -36;
        }

        private int[] refSpeedX = { -1,  1,  0,  0 };
        private int[] refSpeedY = {  0,  0, -1,  1 };

        private int _holeLoop = 0;

        public void ResetHoleLoop()
        {
            Debug.WriteLine("Reset HoleLoop");
            _holeLoop = 0;
        }

        public void Update(GamePadState gamepad, KeyboardState keyboard, GameTime gameTime)
        {
            bool force = (!_inHole) && (_speedX == 0) && (_speedY == 0);
            if (_killed)
            {
                _gameScreen.KillMonster(this, _height, _level);
                return;
            }
            if (_falling)
            {
                _y += _speedY;
                if (_grid.EndFall(_x, _y))
                {
                    // Monster has reached something "hard"
                    int height = (_startFallY - _y) / 12;
                    if (height >= _level)
                    {
                        if ((_cue != null) && _cue.IsPlaying)
                        {
                            _cue.Stop(AudioStopOptions.Immediate);
                            _cue = null;
                        }
                        // Monster is dead
                        _height = height;
                        Die();
                    }
                    else
                    {
                        _falling = false;
                        _speedY = 0;
                        _speedX = 0;
                        _changeDir = true;
                        _inHole = false;
                        Update(gamepad, keyboard, gameTime);
                        return;
                    }
                } else {
                    if (_y % 12 == 9) {
                        _grid.FillHole(_x, _y + 3);
                    }
                }

            }
            else if (!_inHole)
            {
                if (_grid.IsInHole(_x, _y))
                {
                    _x = _grid.SetMonsterInHole(_x, _y, this);
                    _holeLoop = 0;
                    _inHole = true;
                    _sprite = _inHoleSprites[_level - 1];
                    _oldY = _y;
                    _y -= 3;
                    _oldSpeedX = _speedX;
                    _speedX = 0;
                    _speedY = 0;
                    _soundService.PlaySound(Sound.Falling);
                }
                else
                {
                    if (_changeDir)
                    {
                        _speedX = -_speedX;
                        _speedY = -_speedY;
                        _changeDir = false;
                    }
                    if (force || _grid.IsCrossing(_x, _y) || !_grid.CanGo(_x, _y, _speedX, _speedY, true))
                    {
                        bool found = false;
                        // Compute next direction
                        // Special algorithm if last monster : favour "goto to man's level"
                        if (_gameScreen.CurrentMonsterCount == 1)
                        {
                            int index = rnd.Next(4);
                            if (index != 0)
                            {
                                int y = _gameScreen.Man.Y;
                                if ((y > _y) && _grid.CanGo(_x, _y, 0, 1, true))
                                {
                                    found = true;
                                    _speedX = 0;
                                    _speedY = 1;
                                }
                                else if ((y < _y) && _grid.CanGo(_x, _y, 0, -1, true))
                                {
                                    found = true;
                                    _speedX = 0;
                                    _speedY = -1;
                                }
                                else
                                {
                                    int x = _gameScreen.Man.X;
                                    if ((x > _x) && _grid.CanGo(_x, _y, 1, 0, true))
                                    {
                                        found = true;
                                        _speedX = 1;
                                        _speedY = 0;
                                    }
                                    else if ((x < _x) && _grid.CanGo(_x, _y, -1, 0, true))
                                    {
                                        found = true;
                                        _speedX = -1;
                                        _speedY = 0;
                                    }
                                }
                            }
                        }
                        if (!found)
                        {
                            // Normal algorithm : all directions with same probability
                            do
                            {
                                int index = rnd.Next(4);
                                if (_grid.CanGo(_x, _y, refSpeedX[index], refSpeedY[index], true))
                                {
                                    found = true;
                                    _speedX = refSpeedX[index];
                                    _speedY = refSpeedY[index];
                                }
                            } while (!found);
                        }
                    }
                    _x += _speedX;
                    _y += _speedY;
                    _grid.FillPartialHole(_x, _y);
                }
                _sprite.IncrementTexture();
            }
            else
            {
                if (++_holeLoop % 5 == 0)
                {
                    _sprite.IncrementTexture();
                    if (_holeLoop % ExitCycle == 0)
                    {
                        // Dbut de sortie
                        _soundService.PlaySound(Sound.Exit);
                        _grid.SetMonsterInHole(_x, _oldY, null);
                        _sprite = _sprites[_level - 1];
                        _y++;
                        if (_holeLoop == (3*ExitCycle))
                        {
                            // no more hole
                           _grid.FillHole(_x, _y);
                            _inHole = false;
                            _changeDir = false;
                            _speedX = _oldSpeedX;
                            if (_level < 3)
                            {
                                _sprite = _sprites[_level++];
                            }
                            return;
                        }
                    }
                }
            }
            if (_inHole)
            {
                _sprite.OffsetX = Constants.CellWidth / 2;
            }
            else
            {
                _sprite.OffsetX = 0;
            }
            _spriteX = _grid.GetX(_x, _sprite.HorizontalAlignment);
            _spriteY = _grid.GetY(_y);
            _bb.Min = new Vector3((float)_spriteX, (float)_spriteY, 0);
            _bb.Max = new Vector3((float)(_spriteX + _sprite.Width), (float)(_spriteY + _sprite.Height), 0);
        }

        public void Draw(SpriteBatch spriteBatch)
        {
            _sprite.X = _spriteX;
            _sprite.Y = _spriteY;
            _sprite.Draw(spriteBatch);
 
        }

        public bool Collide(Monster other)
        {
            return _bb.Intersects(other._bb);
        }

        public void Fall()
        {
            _cue = _soundService.PlaySound(Sound.Fall);
            _falling = true;
            _startFallY = _y + 3;
            _speedX = 0;
            _speedY = -1;
            _sprite = _sprites[_level - 1];
        }

        public void Die()
        {
            if (!_falling)
            {
                // Another monster fell on it : height = 1
                _height = 1;
            }
            _killed = true;
            _speedY = 0;
        }
    } // class
} // namespace