using System;
using System.Collections.Generic;
using System.IO;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Audio;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.Graphics;
using Microsoft.Xna.Framework.Input;
using Microsoft.Xna.Framework.Storage;

namespace XnaPanic
{

    /// <summary>
    /// This is the main type for your game
    /// </summary>
    public class XnaPanicGame : Microsoft.Xna.Framework.Game, IScoreService
    {
        GraphicsDeviceManager _graphics;
        ContentManager _content;

        SoundManager _soundService;
        SpriteBatch _spriteBatch;
        GameState _gameState;
        NumberTracer _numberTracer;
        MenuScreen _menuScreen;
        GameScreen _gameScreen;

        public IGameScreen CurrentScreen
        {
            get
            {
                switch (_gameState)
                {
                    case GameState.Menu:
                        return _menuScreen;
                    case GameState.Play:
                        return _gameScreen;
                    default:
                        return null;
                }
            }
        }

        public XnaPanicGame()
        {
            _graphics = new GraphicsDeviceManager(this);
            _graphics.PreferredBackBufferFormat = SurfaceFormat.Color;
            _graphics.PreferredBackBufferHeight = Constants.PhysicalHeight * Constants.CellHeight + Constants.MarginTop + Constants.MarginBottom;
            _graphics.PreferredBackBufferWidth = Constants.Width * Constants.CellWidth + Constants.MarginLeft + Constants.MarginRight;
            _graphics.IsFullScreen = false;
            _graphics.ApplyChanges();

            _content = new ContentManager(Services);

            // Get updates at a constant rate
            this.IsFixedTimeStep = true;
            this.TargetElapsedTime = new TimeSpan(0,0,0,0,Constants.TargetElapsedTime);

            _numberTracer = new NumberTracer();
            _soundService = new SoundManager();

            // Register services
            Services.AddService(typeof(IScoreService), this);
            Services.AddService(typeof(INumberTracerService), _numberTracer);
            Services.AddService(typeof(ISoundService), _soundService);

            _menuScreen = new MenuScreen(this);
            _gameScreen = new GameScreen(this);
        }


        /// <summary>
        /// Allows the game to perform any initialization it needs to before starting to run.
        /// This is where it can query for any required services and load any non-graphic
        /// related _content.  Calling base.Initialize will enumerate through any components
        /// and initialize them as well.
        /// </summary>
        protected override void Initialize()
        {
            base.Initialize();
            _soundService.LoadSounds();
            _gameState = GameState.Menu;
            LoadHighScore();
        }


        /// <summary>
        /// Load your _graphics _content.  If loadAllContent is true, you should
        /// load _content from both ResourceManagementMode pools.  Otherwise, just
        /// load ResourceManagementMode.Manual _content.
        /// </summary>
        /// <param name="loadAllContent">Which type of _content to load.</param>
        protected override void LoadGraphicsContent(bool loadAllContent)
        {
            if (loadAllContent)
            {
                _numberTracer.LoadGraphicsContent(_content);
                _menuScreen.LoadGraphicsContent(_content, _graphics.PreferredBackBufferWidth, _graphics.PreferredBackBufferHeight);
                _gameScreen.LoadGraphicsContent(_content, _graphics.PreferredBackBufferWidth, _graphics.PreferredBackBufferHeight);
                _spriteBatch = new SpriteBatch(_graphics.GraphicsDevice);
            }

            // TODO: Load any ResourceManagementMode.Manual _content
        }

        protected override void OnExiting(object sender, EventArgs args)
        {
            SaveHighScore();
            base.OnExiting(sender, args);
        }

        /// <summary>
        /// Unload your _graphics _content.  If unloadAllContent is true, you should
        /// unload _content from both ResourceManagementMode pools.  Otherwise, just
        /// unload ResourceManagementMode.Manual _content.  Manual _content will get
        /// Disposed by the GraphicsDevice during a Reset.
        /// </summary>
        /// <param name="unloadAllContent">Which type of _content to unload.</param>
        protected override void UnloadGraphicsContent(bool unloadAllContent)
        {
            if (unloadAllContent == true)
            {
                _content.Unload();
            }
        }


        /// <summary>
        /// Allows the game to run logic such as updating the world,
        /// checking for collisions, gathering input and playing audio.
        /// </summary>
        /// <param name="gameTime">Provides a snapshot of timing values.</param>
        protected override void Update(GameTime gameTime)
        {
            GamePadState gamepad = GamePad.GetState(PlayerIndex.One);
            KeyboardState keyboard = Keyboard.GetState();

            switch (_gameState)
            {
                case GameState.Menu:
                    _gameState = CurrentScreen.Update(gamepad, keyboard, gameTime);
                    if (_gameState == GameState.Play)
                    {
                        _gameScreen.NewGame();
                    }
                    break;
                case GameState.Play:
                    _gameState = CurrentScreen.Update(gamepad, keyboard, gameTime);
                    break;
                case GameState.Exit:
                    Exit();
                    return;
            }
            _soundService.Update();
            base.Update(gameTime);
        }


        /// <summary>
        /// This is called when the game should draw itself.
        /// </summary>
        /// <param name="gameTime">Provides a snapshot of timing values.</param>
        protected override void Draw(GameTime gameTime)
        {
            _graphics.GraphicsDevice.Clear(Color.Black);

            _spriteBatch.Begin(SpriteBlendMode.AlphaBlend);
            if (CurrentScreen != null)
            {
                CurrentScreen.Draw(_spriteBatch);
            }
            _spriteBatch.End();

            base.Draw(gameTime);
        }


        #region IScoreService Members
        int _bestScore = 0;
        private const string GameName = "XnaPanic";
        private const string FileName = "BestScore";

        int IScoreService.Best
        {
            get
            {
                return _bestScore;
            }
        }
        void IScoreService.GameScore(int value)
        {
            if (_bestScore < value)
            {
                _bestScore = value;
            }
        }

        void LoadHighScore()
        {
            try {
                StorageDevice device = StorageDevice.ShowStorageDeviceGuide();
                using (StorageContainer container = device.OpenContainer(GameName)) {
                    string fullPath = Path.Combine(container.Path, FileName);
                    using (StreamReader reader = File.OpenText(fullPath)) {
                        _bestScore = int.Parse(reader.ReadToEnd());
                    }
                }
            } catch  {
                _bestScore = 0;
            }
        }

        void SaveHighScore()
        {
            try
            {
                StorageDevice device = StorageDevice.ShowStorageDeviceGuide();
                using (StorageContainer container = device.OpenContainer(GameName))
                {
                    string fullPath = Path.Combine(container.Path, FileName);
                    using (StreamWriter writer = File.CreateText(fullPath))
                    {
                        writer.Write(_bestScore.ToString());
                    }
                }
            }
            catch
            {
                // Just fail silently, no big deal
            }
       }
        #endregion

    }
}