#include "inotifywatcher.hpp"

const char* INotifyWatcher::alloc_error_msg =
	"INotify: allocation memoire impossible, arrt notification\n";

const char* INotifyWatcher::error_msg =
	"INotify: erreur, arrt notification\n";

const char* INotifyWatcher::unmount_error_msg =
	"INotify: Le peripherique a t dmont "
	"ou le fichier a t dtruit/ou dplac "
	"ou le point de surveillance a t supprim, "
	"arrt notification.\n"
	"Appuyez sur [return] pour arrter "
	"totalement l'application.\n";


INotifyWatcher::INotifyWatcher(std::string inode,
	int fd, unsigned int events) :
		inode(inode), fd_tube_in(fd), buffer_size(0),
		fd_inotify(-1), fd_watch(-1), fini(false)
{
	fd_inotify = inotify_init ();
	if (fd_inotify < 0)
	{
		// Dans une application relle on lverait probablement
		// une exception un peu plus volue ici ;-)
        throw 1;
	}
	// allouer un tampon minimum pour ne
	// pas avoir de pointeur nul
	buffer.reset(new char[1]);
	//Dfinir le point de surveillance
	fd_watch = inotify_add_watch (fd_inotify,
    		       inode.c_str(), events);
    if(fd_watch < 0)
    {
    	close(fd_inotify);
    	throw 2; // Mme remarque ;-)
    }
    
	// Lancer le thread de surveillance 	
	thread.reset(
		new boost::thread(
			boost::bind(&INotifyWatcher::worker,
				boost::ref(*this))));
				
}

INotifyWatcher::~INotifyWatcher()
{
	destroy_fd();
	thread->join();		
}

void INotifyWatcher::destroy_fd()
{
	boost::try_mutex::scoped_try_lock stl(tmtx);

	if(fd_inotify == -1)
		return;
	fini = true;
	inotify_rm_watch (fd_inotify, fd_watch);
	close(fd_inotify);
	fd_inotify = -1;
	fd_watch = -1;
}

void INotifyWatcher::worker()
{
	int len_read;
	
	while(!fini)
	{
		len_read = read(fd_inotify,
		 	buffer.get(), buffer_size);
		 // Si la mthode destroy_fd est appele
		 // pour arrter, cela dbloque read
		 // donc on arrive l :)
		 if(fini)
		 	break;
		 // sinon, on bosse
		 if(len_read <0)
		 {
		 	write(fd_tube_in, error_msg, strlen(error_msg));
		 	fini = true;
		 	continue;
		 }
		  

		 if(len_read == 0)
		 {
		 	// Ajouter un tampon lmentaire,
		 	// autant que ncessaire
		 	try
		 	{
		 		buffer_size += one_buffer_size;
		 		buffer.reset(new char[buffer_size]);
		 		continue; // => retourner  read
		 	}
		 	catch(std::bad_alloc)
		 	{
		 		write(fd_tube_in, alloc_error_msg,
		 			strlen(alloc_error_msg));
		 		fini = true;
		 		continue; // => arrter le thread
		 	}
		 }
		 
		 int index = 0;
		 while(index < len_read)
		 {
		 	struct inotify_event *event;
		 	event = reinterpret_cast<struct inotify_event *>
		 		(&buffer[index]);
		 		
		 	try
		 	{
		 		std::ostringstream os;
		 		os << std::hex;
		 		os << "inotify: " << inode << " ";
		 		if(event->len)
		 			os << " " << event->name;
		 		os << " " << event->mask << " -> ";
		 		os << mask2text(event->mask);
		 		os << std::endl;
		 		write(fd_tube_in,
		 			os.str().c_str(), os.str().length());
		 		
		 		if( event->mask & IN_UNMOUNT
		 			|| event->mask & IN_IGNORED
		 			|| event->mask & IN_DELETE_SELF)
		 		{
		 			write(fd_tube_in, unmount_error_msg,
		 				strlen(unmount_error_msg));
		 			fini = true; 	// => arrt 
		 			break;			// => quitter le while imbriqu    
		 			// Sans le break, on pompe tous les
		 			// ventuels messages d'erreurs connexes
		 			// avant d'arrter (question de choix...) 
		 		}
		 			
		 		index += event_size + event->len;
		 	}
		 	catch(std::bad_alloc)
		 	{
		 		// on arrive ici si
		 		// std::ostringstream
		 		// lve std::bad_alloc
		 		write(fd_tube_in, alloc_error_msg,
		 			strlen(alloc_error_msg));
		 		fini = true;
		 		continue;
		 	}
		 } //while(index < len_read)	 
	} // while(!fini)
	
	boost::try_mutex::scoped_try_lock local_stl(tmtx, false);
	if(local_stl.try_lock())
		destroy_fd();
}


std::string INotifyWatcher::mask2text(unsigned int mask)
{
	std::ostringstream os ;
	
	if(mask & IN_CREATE)
		os << "IN_CREATE ";
	if(mask & IN_ACCESS)
		os << "IN_ACCESS ";
	if(mask & IN_MODIFY)
		os << "IN_MODIFY ";
	if(mask & IN_ATTRIB)
		os << "IN_ATTRIB ";
	if(mask & IN_CLOSE_WRITE)
		os << "IN_CLOSE_WRITE ";
	if(mask & IN_CLOSE_NOWRITE)
		os << "IN_CLOSE_NOWRITE ";
	if(mask & IN_OPEN)
		os << "IN_OPEN ";
	if(mask & IN_MOVED_FROM)
		os << "IN_MOVED_FROM ";
	if(mask & IN_MOVED_TO)
		os << "IN_MOVED_TO ";
	if(mask & IN_MOVE_SELF)
		os << "IN_MOVE_SELF ";
	if(mask & IN_DELETE)
		os << "IN_DELETE ";
	if(mask & IN_DELETE_SELF)
		os << "IN_DELETE_SELF ";
	if(mask & IN_ISDIR)
		os << "IN_ISDIR ";
	if(mask & IN_ONESHOT)
		os << "IN_ONESHOT ";
	if(mask & IN_IGNORED)
		os << "IN_IGNORED ";
	return os.str();
}
